#! /bin/bash -p
#
# Copyright (c) Omnissa, LLC. All rights reserved.
# This product is protected by copyright and intellectual property laws in the
# United States and other countries as well as by international treaties.
# -- Omnissa Restricted
#
# Omnissa Scanner Redirection and SerialPort Redirection linux client Setup
#

#
# Global variables
#
g_feature_name="Scanner-SerialPort-Common"

#
# print_usage():
# This function prints script usage.
#
print_usage() {
   printf "$0:\nThis script installs/uninstalls ${g_feature_name}."
   printf "Please note that you will require root privileges to execute this script.\n"
   printf "\nUsage:\n"
   if [ "${g_feature_name}" = "Scanner-SerialPort-Common" ]; then
      printf "$0 -i|--install|-u|--uninstall <INIT_DIR> <INII_SCRIPT_DIR> <ServiceName>\n"
   else
      printf "$0 -i|--install|-u|--uninstall <INIT_DIR> <INII_SCRIPT_DIR>\n"
   fi
   printf "Pass -i or --install to install ${g_feature_name} files.\n"
   printf "Pass -u or --uninstall to uninstall ${g_feature_name} files.\n"
   printf "Pass <INII_DIR> to specify the init dir(For example, /etc.\n"
   printf "Pass <INII_SCRIPT_DIR> to specify the init script dir(For example, /etc/init.d).\n"
   if [ "$g_feature_name" = "Scanner-SerialPort-Common" ]; then
      printf "Pass <ServiceName> to specify the service name(For example, ftscanhv|ftsprhv).\n"
   fi
}

#
# register_service():
# This function register specified service for Debian-like Linux.
#
register_service() {
   local service_name=$1
   local script_path=$2
   local init_dir=$3
   if [ -f /usr/sbin/update-rc.d ] || [ -f /usr/bin/update-rc.d ]; then
      update-rc.d ${service_name} remove || true
   fi
   run_levels="2 3 5"
   for run_level in ${run_levels}; do
      ln -sf $script_path "${init_dir}/rc${run_level}.d/K08${service_name}"
      ln -sf $script_path "${init_dir}/rc${run_level}.d/S50${service_name}"
   done
}

#
# install_service():
# This function installs the specified service.
#
install_service() {
   local init_dir=$1
   local init_script_dir=$2
   local service_name=$3
   local PERM_BIN=555
   local PERM_DATA=444
   local PERM_CONF=644
   src=$(dirname "$(readlink -f "$0")")
   chmod ${PERM_BIN} /usr/lib/omnissa/horizon/bin/ft* || true
   chmod ${PERM_BIN} /usr/lib/omnissa/rdpvcbridge/ft* || true
   chmod ${PERM_BIN} /usr/lib/omnissa/horizon/scannerSerialPortCommon/* || true
   chmod ${PERM_CONF}  /etc/omnissa/ft* || true
   if [ -d "/usr/lib/omnissa/horizon/scannerRedirection" ]; then
      chmod ${PERM_DATA} /usr/share/doc/omnissa-horizon-client/scannerClient/* > /dev/null 2>&1 || true
      chmod ${PERM_BIN} /usr/lib/omnissa/horizon/scannerRedirection/* || true
   fi
   if [ -d "/usr/lib/omnissa/horizon/serialPortRedirection" ]; then
      chmod ${PERM_DATA} /usr/share/doc/omnissa-horizon-client/serialPortClient/* > /dev/null 2>&1 || true
      chmod ${PERM_BIN} /usr/lib/omnissa/horizon/serialPortRedirection/* || true
   fi
   # For Ubuntu
   if [ -f /usr/sbin/update-rc.d ] || [ -f /usr/bin/update-rc.d ]; then
      ln -sf ${src}/${service_name}.debian ${init_script_dir}/${service_name} || true
      register_service ${service_name} "${init_script_dir}/${service_name}" ${init_dir}
      update-rc.d ${service_name} defaults || true
      if ! ${init_script_dir}/${service_name} start > /dev/null 2>&1; then
         printf "Warning: The daemon for ${g_feature_name} fails to start!"
      fi
   # For RHEL<9.0
   elif [ -f /usr/sbin/chkconfig ] || [ -f /usr/bin/chkconfig ]; then
      ln -sf ${src}/${service_name}.redhat ${init_script_dir}/${service_name} || true
      register_service ${service_name} "${init_script_dir}/${service_name}" ${init_dir}
      if ! ${init_script_dir}/${service_name} start > /dev/null; then
         printf "Warning: The daemon for ${g_feature_name} fails to start!"
      fi
   # For RHEL>=9.0, only exists systemctl
   elif [ -f /usr/sbin/systemctl ] || [ -f /usr/bin/systemctl ]; then
      cp -f ${src}/${service_name}.service /etc/systemd/system/${service_name}.service || true
      systemctl stop ${service_name} > /dev/null || true
      systemctl disable ${service_name} --now > /dev/null 2>&1 || true
      chcon -u system_u -r object_r -t bin_t /usr/lib/omnissa/horizon/bin/${service_name}d
      systemctl daemon-reload > /dev/null 2>&1 || true
      systemctl enable ${service_name} > /dev/null 2>&1 || true
      systemctl start ${service_name} > /dev/null 2>&1 || true
      if ! pgrep -f ${service_name}d > /dev/null; then
         printf "Warning: The daemon for ${g_feature_name} fails to start!"
      fi
   else
      printf "Error: update-rc.d, chkconfig & systemctl are not installed, failed to config ${g_feature_name}!"
   fi
}

#
# uninstall_service():
# This function uninstalls the specified service.
#
uninstall_service() {
   local init_dir=$1
   local init_script_dir=$2
   local service_name=$3
   if [ -f /usr/sbin/update-rc.d ] || [ -f /usr/bin/update-rc.d ]; then
      ${init_script_dir}/${service_name} stop || true
      update-rc.d -f ${service_name} remove || true
      target_symbolic_file=${init_script_dir}/${service_name}
      rm -rf ${target_symbolic_file} || true
   elif [ -f /usr/sbin/chkconfig ] || [ -f /usr/bin/chkconfig ]; then
      ${init_script_dir}/${service_name} stop >/dev/null || true
      target_symbolic_file=${init_script_dir}/${service_name}
      rm -rf ${target_symbolic_file} || true
      run_levels="2 3 5"
      for run_level in ${run_levels}; do
         rm -rf ${init_dir}/rc${run_level}.d/*${service_name}*
      done
   elif [ -f /usr/sbin/systemctl ] || [ -f /usr/bin/systemctl ]; then
      systemctl stop ${service_name} || true
      systemctl disable ${service_name} > /dev/null 2>&1 || true
      target_symbolic_file=/etc/systemd/system/multi-user.target.wants/${service_name}
      rm -rf ${target_symbolic_file} || true
      rm -rf /etc/systemd/system/${service_name}.service || true
   else
      printf "Warning: Not found update-rc.d, chkconfig or systemctl. Failed to stop ${g_feature_name} services"
   fi
}

main(){
   if [[ $0 == *"serialport-redirection-setup"* ]]; then
      g_feature_name="SerialPort Redirection"
   elif [[ $0 == *"scanner-redirection-setup"* ]]; then
      g_feature_name="Scanner Redirection"
   fi
   if [ "$#" -lt "4" ] ; then
      printf "No enough command line options specified!\n"
      print_usage
      exit 0
   fi

   #
   # Root privileges required to execute this script.
   #
   if [ ! $EUID = "0" ]; then
      printf "Root privileges required!\n"
      exit 1
   fi

   opt=$1
   init_dir=$2
   init_script_dir=$3
   service_name=$4

   case $opt in
      -u|--uninstall)
         uninstall_service ${init_dir} ${init_script_dir} ${service_name}
         exit
      ;;

      -i|--install)
         install_service ${init_dir} ${init_script_dir} ${service_name}
         exit
      ;;
      *)
         printf "Unrecognized command line option!\n"
         print_usage
         exit
      ;;
   esac
}
