/*
 * Copyright (c) 2024 Omnissa, LLC. All rights reserved.
 * This product is protected by copyright and intellectual property laws in the
 * United States and other countries as well as by international treaties.
 * -- Omnissa Public
 */

/*
 * viewusblib.h
 *
 * viewusb API
 */


#ifndef __VIEWUSBLIB_H
#define __VIEWUSBLIB_H


#ifdef __cplusplus
extern "C" {
#endif


#include <stdarg.h>


#ifndef uint
typedef unsigned int uint;
#endif /* uint */


/*
 * Generic ViewUsb handle plus flavours
 */
typedef void *ViewUsb_Handle;
typedef void *ViewUsb_Client;
typedef void *ViewUsb_Desktop;
typedef void *ViewUsb_Enumeration;

typedef long long ViewUsb_DeviceID;

#define ViewUsb_True 1
#define ViewUsb_False 0

/*
 * Function status return values (success plus various failure modes)
 */
typedef enum {
   ViewUsbStatus_Success = 0,
   ViewUsbStatus_BadArguments = 1,
   ViewUsbStatus_BadClient = 2,
   ViewUsbStatus_ResourcesExhausted = 3,
   ViewUsbStatus_NotHeld = 4,
   ViewUsbStatus_EndOfEnumeration = 5,
   ViewUsbStatus_BadHandle = 6,
   ViewUsbStatus_OpenChannelNotPending = 7,
   ViewUsbStatus_UsbdCommsFailure = 8,
   ViewUsbStatus_ChannelNotOpen = 9,
   ViewUsbStatus_DesktopInUse = 10,
   ViewUsbStatus_BadDesktopHandle = 11,
   ViewUsbStatus_ConnectFailure = 12,
   ViewUsbStatus_DisconnectFailure = 13,
   ViewUsbStatus_BadDeviceID = 14,
   ViewUsbStatus_ProtocolFailure = 15,
   ViewUsbStatus_USBStateUnavailable = 16,
   ViewUsbStatus_USBStateTempUnavailable = 17,
   ViewUsbStatus_AlreadyInitialized = 18,
   ViewUsbStatus_RegistrationFailure = 19,
   ViewUsbStatus_NotRegistered = 20,
   ViewUsbStatus_NotAvailable = 21,
   ViewUsbStatus_StatisticsGatheringFailure = 22,
   ViewUsbStatus_UemNotReported = 23,
   ViewUsbStatus_UemNotInstalled = 24,
   ViewUsbStatus_ClientNotExisting = 25,
} ViewUsbStatus;

/*
 * Size of integer count (number of enumerated devices)
 */
#define VIEWUSB_ENUM_HEADER_SZ (sizeof(int))

/*
 * Size of enumeration entry otw, excluding strings (See otw below)
 */
#define VIEWUSB_ENUM_ENTRY_BASE_SZ ((2 * sizeof(long long)) + (7 * sizeof(int)))

typedef struct viewusb_enumerationdata {
   int magic;
   char *name;                          // otw
   char *desktop;                       // otw
   char *serialNum;                     // otw
   ViewUsb_DeviceID id;                 // otw
   int present;                         // otw
   int shared;                          // otw
   unsigned int family;                 // otw
   unsigned long long physpath;         // otw
   unsigned int quirks;                 // otw
   int reconnecting;                    // otw
   int inFilterList;                    // otw
   int inDisableBufferWithoutAudioList; // otw
} *ViewUsb_EnumerationData;

#define VIEWUSB_ENUM_UNSHARED 0
#define VIEWUSB_ENUM_SHARED 1
#define VIEWUSB_ENUM_SHARING 2
#define VIEWUSB_ENUM_UNSHARING 3

/*
 * List of callbacks, callback data structures and an opaque callback data type
 */
typedef enum {
   VIEWUSB_CB_NULL = 0,
   VIEWUSB_CB_OPENCHANNEL,
   VIEWUSB_CB_CONNECTDEVICE,
   VIEWUSB_CB_ERROR,
   VIEWUSB_CB_CEIPDATA,
   VIEWUSB_CB_LAST,
} ViewUsb_CB;

typedef struct {
   ViewUsb_Desktop desktopHandle;
   int usbAvailable;
   int uemTimeouts; // Deprecated field, always return 0 in callback
} ViewUsb_CB_Data_OpenChannel;

typedef struct {
   ViewUsb_Desktop desktopHandle;
   ViewUsb_DeviceID id;
   int autoconnect;
   int connected;
} ViewUsb_CB_Data_ConnectDevice;

typedef struct {
   ViewUsb_Desktop desktopHandle;
   ViewUsb_DeviceID id;
   int msgId;
   char *msgString;
} ViewUsb_CB_Data_Error;

typedef struct {
   ViewUsb_Desktop desktopHandle;
   char *data;
   unsigned int dataSize;
} ViewUsb_CB_Data_CEIPData;

typedef void *ViewUsb_CB_Data;

/*
 * Public logging levels.
 */
typedef enum {
   ViewUsb_LogLevel_Trace = 0,
   ViewUsb_LogLevel_Debug = 1,
   ViewUsb_LogLevel_Info = 2,
   ViewUsb_LogLevel_Error = 4,
} ViewUsb_LogLevel;

/*
 * View USB API calls
 */
ViewUsbStatus ViewUsb_InitLib(void *(*)(ViewUsb_CB, ViewUsb_CB_Data *), ViewUsb_Client *,
                              const char *);
ViewUsbStatus ViewUsb_OpenDesktop(ViewUsb_Client, ViewUsb_Desktop *, const char *, const char *,
                                  int, const char *, const char *, const char *, const char *, int,
                                  const char *);
ViewUsbStatus ViewUsb_CloseDesktop(ViewUsb_Desktop *, const char *);
#define VIEWUSB_USE_DEFAULT_PORT 0
ViewUsbStatus ViewUsb_OpenChannel(ViewUsb_Desktop); /* -> Callback */
ViewUsbStatus ViewUsb_Enumerate(ViewUsb_Desktop, ViewUsb_Enumeration *, const char *);
ViewUsbStatus ViewUsb_EnumGetFirst(ViewUsb_Enumeration, ViewUsb_EnumerationData *);
ViewUsbStatus ViewUsb_EnumGetNext(ViewUsb_Enumeration, ViewUsb_EnumerationData *);
ViewUsbStatus ViewUsb_FreeEnumData(ViewUsb_EnumerationData *);
ViewUsbStatus ViewUsb_EnumCount(ViewUsb_Enumeration, uint *);
ViewUsbStatus ViewUsb_ConnectDevice(ViewUsb_Desktop, ViewUsb_DeviceID, int); /* -> Callback */
ViewUsbStatus ViewUsb_DisconnectDevice(ViewUsb_DeviceID);
ViewUsbStatus ViewUsb_SetFocus(ViewUsb_Desktop);
ViewUsbStatus ViewUsb_ClearFocus(ViewUsb_Desktop);
ViewUsbStatus ViewUsb_Hold(ViewUsb_Handle); /* Increment reference count on the specified handle */
ViewUsbStatus ViewUsb_Release(
   ViewUsb_Handle); /* Decrement reference count on handle, tidy up resources upon last free */
int ViewUsb_USBAvailable(ViewUsb_Desktop);
int ViewUsb_USBUemState(
   ViewUsb_Desktop); /* Deprecated API, always return ViewUsbStatus_UemNotInstalled */
const char *ViewUsb_StatusToString(ViewUsbStatus);
ViewUsbStatus ViewUsb_RegisterLogger(void (*)(ViewUsb_LogLevel, const char *, va_list));

/*
 * ViewUSB CEIP API calls
 *
 * Currently implemented in viewusblib and horizon-usbd for convenience,
 * however it may prove sensible to split out into a common CEIP service
 * to both support multiple data providers (of which ViewUSB is one example)
 * per CEIP client, and to provide CEIP data persistence.
 *
 * char ** is a pointer to an XML string containing the CEIP data.
 * When requesting data the pointer should point to a NULL pointer
 * (no data allocated).
 */
ViewUsbStatus ViewUsb_RegisterCEIPClientApplication(int *, const char *);
ViewUsbStatus ViewUsb_RequestCEIPData(char **, size_t *, const char *);
ViewUsbStatus ViewUsb_ResetCEIPData(int, const char *);

/*
 * ViewUSB Exclude Family calls
 *
 * Use for usb storage over CDR feature.
 */
ViewUsbStatus ViewUsb_AddExcludeFamily(ViewUsb_Desktop, unsigned int);
ViewUsbStatus ViewUsb_RemoveExcludeFamily(ViewUsb_Desktop, unsigned int);

/*
 * ViewUSB custom auto connect calls
 *
 * Use for custom auto connect device list feature.
 */
ViewUsbStatus ViewUsb_ResetAutoConnectInsertDevice(ViewUsb_Desktop);
ViewUsbStatus ViewUsb_ResetAutoConnectStartupDevice(ViewUsb_Desktop);
ViewUsbStatus ViewUsb_AddAutoConnectInsertDevice(ViewUsb_Desktop, ViewUsb_DeviceID);
ViewUsbStatus ViewUsb_AddAutoConnectStartupDevice(ViewUsb_Desktop, ViewUsb_DeviceID);
ViewUsbStatus ViewUsb_RemoveAutoConnectInsertDevice(ViewUsb_Desktop, ViewUsb_DeviceID);
ViewUsbStatus ViewUsb_RemoveAutoConnectStartupDevice(ViewUsb_Desktop, ViewUsb_DeviceID);

#ifdef __cplusplus
}
#endif /* extern "C" */

#endif /* __VIEWUSBLIB_H */
